use std::collections::BTreeMap;

use anyhow::{anyhow, Context, Result};
use chrono::NaiveDate;

const DATE_FORMAT: &str = "%Y-%m-%d";

pub struct SummaryFile {
    summaries: BTreeMap<NaiveDate, String>
}

impl SummaryFile {
    pub fn empty() -> Self {
        Self {
            summaries: BTreeMap::new()
        }
    }

    pub fn read(string: &str) -> Result<Self> {
        let mut s = Self::empty();
        s.deserialize(string).context("failed to deserialize summary file")?;
        Ok(s)
    }

    fn deserialize(&mut self, s: &str) -> Result<()> {
        for (i, line) in s.split('\n').enumerate() {
            if line.is_empty() { continue; }

            let splits = line.splitn(2, ':').collect::<Vec<_>>();
            if splits.len() != 2 {
                return Err(anyhow!("line {i} does not contain a colon"))
            }

            let date = NaiveDate::parse_from_str(splits[0].trim(), DATE_FORMAT)
                .with_context(|| format!("failed to parse date on line {i}"))?;

            let summary = splits[1].trim().to_owned();

            self.summaries.insert(date, summary);
        }

        Ok(())
    }

    pub fn serialize(&self) -> String {
        self.summaries.iter()
            .map(|(date, summary)| format!("{}: {summary}", date.format(DATE_FORMAT)))
            .intersperse("\n".to_string())
            .collect()
    }

    pub fn get(&self, date: NaiveDate) -> Option<String> {
        self.summaries.get(&date).cloned()
    }

    pub fn set(&mut self, date: NaiveDate, summary: String) {
        self.summaries.insert(date, summary);
    }

}
