use std::{fs, path::{Path, PathBuf}};

use anyhow::{anyhow, Context, Result};
use chrono::NaiveDate;
use config::Config;
use gpg::{decrypt, encrypt};
use summary::SummaryFile;

const FILE_CONFIG: &str = ".config";
const FILE_SANITY: &str = ".sanity";
const FILE_SSH_KEY: &str = ".ssh-key";
const FILE_SUMMARIES: &str = "summaries";

const SANITY_CONTENT: &str = "yes I am sane the user uses the same password thank god";

mod gpg;
pub mod config;
pub mod summary;

/// a locked version of the repository to read the config from
pub struct LockedRepository {
    pub path: PathBuf,
    pub config: Config
}

impl LockedRepository {
    /// opens the repository by reading the config
    pub fn open(path: &Path) -> Result<Self> {
        Ok(Self {
            path: path.to_owned(),
            config: Self::read_config(path)?.unwrap_or_default()
        })
    }

    /// read the repository config file
    fn read_config(path: &Path) -> Result<Option<Config>> {
        let file = path.join(FILE_CONFIG);
        if !file.exists() { return Ok(None); }

        let string = fs::read_to_string(file)
            .context("repository config file cannot be read")?;

        toml::from_str(&string).context("failed to parse repository config")
    }

    /// convert this repository into an unlocked one
    /// this now takes a reference so we can have multiple tries at unlocking (at the cost of cloning the config, oh well)
    pub fn unlock(&self, passphrase: &str) -> Result<UnlockedRepository> {
        let unlocked = UnlockedRepository {
            path: self.path.clone(),
            passphrase: passphrase.to_owned(),
            config: self.config.clone()
        };

        unlocked.check_sanity()?;

        Ok(unlocked)
    }
}

/// an unlocked repository with an internal passphrase
pub struct UnlockedRepository {
    passphrase: String,
    pub path: PathBuf,
    pub config: Config
}

impl UnlockedRepository {

    /// does a given file exist inside the repo
    fn exists_file(&self, file: &str) -> bool {
        let file = self.path.join(file);

        file.exists() && file.is_file()
    }

    /// read a given file from the repo
    fn read_file(&self, file: &str) -> anyhow::Result<String> {
        let file = self.path.join(file);

        let encrypted = fs::read(file)
            .context("failed to read file from repository")?;

        decrypt(&encrypted, &self.passphrase)
            .context("decryption failed")
    }

    /// write to a given file in the repo
    fn write_file(&self, file: &str, content: &str) -> anyhow::Result<()> {
        let file = self.path.join(file);

        let encrypted = encrypt(content, &self.passphrase, &self.config.encryption.cipher, self.config.encryption.armored)
            .context("encryption failed")?;

        fs::write(file, encrypted)
            .context("failed to write file to repository")
    }

    /// reads the ssh key from file
    pub fn read_ssh_key(&self) -> anyhow::Result<String> {
        self.read_file(FILE_SSH_KEY)
            .context("failed to read ssh key file")
    }

    /// performs a sanity check if required
    fn check_sanity(&self) -> anyhow::Result<()> {
        if !self.config.encryption.password_sanity { return Ok(()); }

        if self.exists_file(FILE_SANITY) {
            let content = self.read_file(FILE_SANITY)
                .context("passphrase sanity check failed")?;

            if content != SANITY_CONTENT {
                return Err(anyhow!("sanity check almost succeeded, but message was insane: {content}"))
            }
        } else {
            println!("no sanity file found, creating now");

            self.write_file(FILE_SANITY, SANITY_CONTENT)
                .context("passphrase sanity initialization failed")?;
        }

        Ok(())
    }

    /// returns the path to the file of an entry
    pub fn file_for_entry(&self, date: NaiveDate) -> PathBuf {
        self.path.join(date.format(&self.config.date_format).to_string())
    }

    /// returns the size of the entry _on the filesystem_ in bytes
    pub fn size_of_entry(&self, date: NaiveDate) -> Result<u64> {
        fs::metadata(self.file_for_entry(date))
            .map(|meta| meta.len())
            .context("failed to read metadata for file")
    }

    /// returns whether a specific journal entry exists
    pub fn exists_entry(&self, date: NaiveDate) -> bool {
        self.exists_file(&date.format(&self.config.date_format).to_string())
    }

    /// reads a specific journal entry from the repository
    pub fn read_entry(&self, date: NaiveDate) -> Result<String> {
        self.read_file(&date.format(&self.config.date_format).to_string())
    }

    /// writes a specific journal entry to the repository
    pub fn write_entry(&self, date: NaiveDate, content: &str) -> Result<()> {
        self.write_file(&date.format(&self.config.date_format).to_string(), content)
    }

    /// lists all journal entries in the repository
    pub fn list_entries(&self) -> Result<Vec<NaiveDate>> {
        Ok(
            fs::read_dir(&self.path).context("failed to read repository directory")?
                .filter_map(|result| result.ok())
                .filter_map(|entry| {
                    NaiveDate::parse_from_str(&entry.file_name().to_string_lossy(), &self.config.date_format).ok()
                }).collect()
        )
    }

    /// reads the summaries from disk
    pub fn read_summaries(&self) -> Result<SummaryFile> {
        if self.exists_file(FILE_SUMMARIES) {
            self.read_file(FILE_SUMMARIES)
                .and_then(|file| SummaryFile::read(&file))
        } else {
            Ok(SummaryFile::empty())
        }
    }

    /// writes the summaries back to disk
    pub fn write_summaries(&self, summaries: &SummaryFile) -> Result<()> {
        self.write_file(FILE_SUMMARIES, &summaries.serialize())
    }

    /// returns the path for the summaries
    pub fn file_for_summaries(&self) -> PathBuf {
        self.path.join(FILE_SUMMARIES)
    }
}
