use std::{fs::File, io::Write, os::fd::{FromRawFd, IntoRawFd}, process::{Command, Stdio}};

use anyhow::{anyhow, Context, Result};
use command_fds::{CommandFdExt, FdMapping};
use nix::unistd;

#[derive(Clone, Copy, PartialEq)]
enum GpgAction {
    Encrypt,
    Decrypt
}

/// constructs a gpg command with the relevant flags
fn gpg_command(armor: bool, action: GpgAction, cipher: &str) -> Command {
    let mut command = Command::new("gpg");

    // set action for gpg
    command.arg(match action {
        GpgAction::Encrypt => "--symmetric",
        GpgAction::Decrypt => "--decrypt",
    });

    command.arg("--batch"); // makes sure gpg does not enter interactive mode
    command.arg("--no-symkey-cache"); // make double sure we don't use the key cache
    command.arg("--quiet"); // removes unneeded output

    if action == GpgAction::Encrypt && armor {
        command.arg("--armor"); // makes output ascii armored
    };

    if action == GpgAction::Encrypt {
        command.arg("--cipher-algo").arg(cipher); // set cipher when encrypting
    }

    command.arg("--passphrase-fd").arg("3"); // read password from fd 3

    command
}

/// runs the given command with input on the stdin and returns the output. handles errors correctly
fn gpg_run(mut command: Command, passphrase: &str, input: &[u8]) -> Result<Vec<u8>> {

    // create pipe for passphrase
    let (pass_read, pass_write) = unistd::pipe()
        .context("failed to create pipe for passphrase")?;

    // write passphrase to pipe
    let mut file = unsafe { File::from_raw_fd(pass_write.into_raw_fd()) };
    file.write_all(passphrase.as_bytes()).context("failed to write passphrase to pipe")?;
    file.flush().context("failed to flush passphrase pipe")?;
    drop(file); // close the pipe

    let mut child = command
        .stdin(Stdio::piped())
        .stdout(Stdio::piped())
        .stderr(Stdio::piped())
        .fd_mappings(vec![FdMapping { child_fd: 3, parent_fd: pass_read}])
        .context("failed to map pipe for passphrase")?
        .spawn()
        .context("failed to spawn gpg")?;

    // write data to stdin
    let mut stdin = child.stdin.take().expect("stdin is piped");
    stdin.write_all(input).context("failed to write to gpg stdin")?;
    stdin.flush().context("failed to flush gpg stdin")?;
    drop(stdin); // close the stdin

    // wait for output
    let output = child.wait_with_output().context("failed to wait for gpg output")?;

    if output.status.success() {
        Ok(output.stdout)
    } else {
        let mut error = String::from_utf8_lossy(&output.stderr).trim().to_owned();

        if error == "gpg: decryption failed: Bad session key" {
            error = "passphrase did not match".to_owned();
        }

        Err(anyhow!("{error}"))
    }
}

/// encrypts plain with the given passphrase and cipher, can be armored
pub fn encrypt(plain: &str, passphrase: &str, cipher: &str, armor: bool) -> Result<Vec<u8>> {
    gpg_run(
        gpg_command(armor, GpgAction::Encrypt, cipher),
        passphrase,
        plain.as_bytes()
    )
}

/// decrypts the provide cipher text with the given passphrase (text can be any cipher and/or armored)
pub fn decrypt(ciphertext: &[u8], passphrase: &str) -> Result<String> {
    gpg_run(
        gpg_command(false, GpgAction::Decrypt, ""),
        passphrase,
        ciphertext
    ).and_then(|bytes|
        String::from_utf8(bytes)
            .context("failed to decode plaintext as utf-8")
    )
}
