use chrono::{NaiveDate, Utc};
use serde::Deserialize;

#[derive(Deserialize, Debug, Clone)]
#[serde(default)]
pub struct Config {
    /// date format for the notes file name
    pub date_format: String,
    /// hour of the daybreak
    pub daybreak: u32,

    /// editor config
    pub editor: ConfigEditor,
    /// encryption config
    pub encryption: ConfigEncryption,
    /// git config
    pub git: ConfigGit
}

impl Default for Config {
    fn default() -> Self {
        Self {
            date_format: "%Y-%m-%d".to_string(),
            daybreak: 0,
            editor: ConfigEditor::default(),
            encryption: ConfigEncryption::default(),
            git: ConfigGit::default()
        }
    }
}

#[derive(Deserialize, Debug, Clone)]
#[serde(default)]
pub struct ConfigEncryption {
    /// ascii-armor encrypted files
    pub armored: bool,
    /// gpg cipher to use for encryption
    pub cipher: String,
    /// check for password sanity
    pub password_sanity: bool,
}

impl Default for ConfigEncryption {
    fn default() -> Self {
        Self {
            armored: true,
            cipher: "AES256".to_string(),
            password_sanity: true,
        }
    }
}

#[derive(Deserialize, Debug, Clone)]
#[serde(default)]
pub struct ConfigGit {
    /// enable git support
    pub enabled: bool,
    /// use ssh key support
    pub ssh: bool,
    /// force offline mode at all times
    pub force_offline: bool,
    /// date since git has been used on this repo
    #[serde(with = "date_format")]
    pub since: NaiveDate,
    /// message styling
    pub message: ConfigGitMessage,
}

impl Default for ConfigGit {
    fn default() -> Self {
        Self {
            enabled: false,
            ssh: true,
            force_offline: false,
            since: NaiveDate::MIN,
            message: ConfigGitMessage::default()
        }
    }
}

#[derive(Deserialize, Debug, Clone)]
#[serde(default)]
pub struct ConfigGitMessage {
    /// verb used for new files
    pub verb_new: String,
    /// verb used for updating files
    pub verb_update: String,
    /// tag added at end for forced upates
    pub tag_force: String,
    /// date format for things
    pub date_format: String,
    /// string used for named notes
    pub named: String
}

impl Default for ConfigGitMessage {
    fn default() -> Self {
        Self {
            verb_new: "new".to_owned(),
            verb_update: "update".to_owned(),
            tag_force: "(forced)".to_owned(),
            date_format: "%d.%m.%y".to_owned(),
            named: "note entry".to_owned(),
        }
    }
}

#[derive(Deserialize, Debug, Clone)]
#[serde(default)]
pub struct ConfigEditor {
    /// command to run as editor
    pub command: String,
    /// arguments to pass to the command before the filename
    pub arguments: Vec<String>,
    /// prompt to be shown instead on a `--void` insertion
    pub void: String,
}

impl Default for ConfigEditor {
    fn default() -> Self {
        Self {
            command: "vim".to_owned(),
            arguments: vec![],
            void: "enter stuff to append to your entry (submit empty to exit)".to_string(),
        }
    }
}


mod date_format {
    use chrono::{NaiveDate};
    use serde::{self, Deserialize, Serializer, Deserializer};

    const FORMAT: &str = "%Y-%m-%d";

    pub fn serialize<S>( date: &NaiveDate, serializer: S) -> Result<S::Ok, S::Error> where S: Serializer {
        let s = format!("{}", date.format(FORMAT));
        serializer.serialize_str(&s)
    }

    pub fn deserialize<'de, D>(deserializer: D) -> Result<NaiveDate, D::Error> where D: Deserializer<'de> {
        let s = String::deserialize(deserializer)?;
        NaiveDate::parse_from_str(&s, FORMAT).map_err(serde::de::Error::custom)
    }
}
