#![feature(iter_intersperse)]
#![allow(unused)]

use std::{path::PathBuf, thread::sleep, time::Duration};

use anyhow::{anyhow, Context, Result};
use browse::list::{BrowseEntry, EntryHistory};
use chrono::{Days, Local, NaiveDate, Timelike};
use clap::Parser;
use command::{Args, Command};
use config::GlobalConfig;
use git::{construct_commit_message, Git};
use input::{date::{is_absolute, parse_absolute}, read_pass};
use repository::{LockedRepository, UnlockedRepository};

pub mod config;

mod command;
mod repository;
mod editor;
mod git;
mod input;

mod browse;

fn main() -> Result<()> {
    let args = Args::parse();
    let mut now = Local::now().naive_local().date();

    let repo_path = args.repository
        .or_else(|| std::env::var("ME_REPOSITORY").ok())
        .or_else(|| GlobalConfig::read().map(|g| g.repository))
        .map(PathBuf::from)
        .context("no repository is specified, neither in command nor config")?;

    if !repo_path.exists() || !repo_path.is_dir() {
        return Err(anyhow!("specified repository does not exist or is not a directory"));
    }

    let repo = LockedRepository::open(&repo_path)
            .context("failed to open repository")?;

    let input = args.date.map(|vec| vec.join(" "));

    let mut date = input.as_ref()
        .map(|a| parse_absolute(a))
        .unwrap_or_else(|| Ok(now))?;

    // go back one day if early in the morning (we only perform this on relative dates)
    let relative = input.as_ref().map(|a| !is_absolute(a)).unwrap_or(true);

    if Local::now().hour() < repo.config.daybreak && relative {
        now = now.checked_sub_days(Days::new(1)).context("daybreak adjustment failed")?;
        date = date.checked_sub_days(Days::new(1)).context("daybreak adjustment failed")?;
    }

    // prevent editing of future dates
    if date > now && !args.force {
        return Err(anyhow!("you cannot journal what still lies ahead"));
    }

    // unlock repository
    let mut tries = 0;
    let repo = loop {
        tries += 1;

        match repo.unlock(&read_pass("please enter your passphrase")?) {
            Ok(repo) => break repo,
            Err(err) => {
                // yeah, this won't do anything because they can use gpg directly or just kill the process but why not annoy the user a bit
                sleep(Duration::from_millis(tries * 500));

                println!("failed to unlock repository: {err}");

                if tries >= 5 {
                    return Err(anyhow!("failed to unlock repository {tries} times in a row, maybe do something else?"))
                }
            },
        }
    };

    // update repository
    let git = if repo.config.git.enabled && !args.gitless {
        let instance = Git::new(
            &repo.path,
            if repo.config.git.ssh { Some(repo.read_ssh_key()?) } else { None },
            repo.config.git.force_offline || args.offline
        )?;

        instance.pull()
            .context("failed to update repository")?;

        Some(instance)
    } else { None };

    if matches!(args.command, Some(Command::Browse)) {
        let mut dates = repo.list_entries().context("failed to list all entries")?;
        dates.sort();

        let create_entries = || -> Result<Vec<BrowseEntry>> {
            let summaries = repo.read_summaries().context("failed to read summaries")?;
            let history = git.as_ref().map(|g| g.history(&repo.config.git.message)).transpose().context("failed to read git history")?;

            Ok(
                dates.iter()
                    .map(|date|
                        BrowseEntry {
                            date: *date,
                            summary: summaries.get(*date),
                            history: history.as_ref()
                                .filter(|_| date >= &repo.config.git.since) // only show history for entries after git since
                                .and_then(|map| map.get(&date.format(&repo.config.date_format).to_string()))
                                .map(|history| {
                                    EntryHistory {
                                        changes: history.len(),
                                        force: history.iter().any(|c| c.force),
                                        time: history.iter().filter(|c| !c.force).map(|c| c.time).max().map(|a| a.time())
                                    }
                                }),
                            size: repo.size_of_entry(*date).ok()
                    })
                    .collect()
            )
        };

        let mut current_date = now;
        let mut entries = create_entries()?;

        loop {
            match browse::select(&entries, Some(current_date)).context("failed to show select ui")? {
                Some(date) => {
                    open(&repo, &git, date, args.force, date < now && args.force, args.void)?;
                    current_date = date;
                },
                None => break,
            };

            // we need to re-read summaries and history if we change stuff
            if args.force {
                entries = create_entries()
                    .context("failed to recreate entry list after editing")?;
            }
        }

    } else {
        // normal editing
        open(&repo, &git, date, date >= now || args.force, date < now && args.force, args.void)?;
    }

    Ok(())
}

const SUMMARY_PREFIX: &str = "?? summary:";

fn open(repo: &UnlockedRepository, git: &Option<Git>, date: NaiveDate, edit: bool, force: bool, void: bool) -> Result<()> {
    let mut summaries = repo.read_summaries()
        .context("failed to read summaries")?;

    let exists = repo.exists_entry(date);
    let name = date.format("%d.%m.%y").to_string();

    let before = if exists {
        repo.read_entry(date).context("failed to read journal entry")?
    } else {
        "".to_owned()
    };
    let before_summary = summaries.get(date).unwrap_or_default();

    let content = format!("{SUMMARY_PREFIX} {before_summary}\n\n{before}");

    if edit {
        let result = if void {
            let mut buffer = content.clone();

            loop {
                let append = input::read_pass(&repo.config.editor.void)?;

                if (append.trim().is_empty()) {
                    break;
                }

                // push a couple of newlines so the paragraphs are separated
                buffer.push('\n');
                buffer.push_str(&append);
                buffer.push('\n');
            }

            buffer
        } else {
            editor::edit(&content, &name, &repo.config.editor)?
        };

        // FIXME: for now we only check the first line and strip it if nessecary
        let (mut after, summary) = if let Some(things) = result.strip_prefix(SUMMARY_PREFIX) {
            let mut splits = things.splitn(2, '\n');

            let summary = splits.next().context("no summary found")?.trim().to_owned();
            let after = splits.next().context("no body found")?.trim().to_owned();

            (after, Some(summary))

        } else {
            (result, None)
        };

        // make sure files end with something
        if !after.ends_with('\n') {
            after.push('\n');
        }

        let mut changes = vec![];

        if before.trim() != after.trim() {
            repo.write_entry(date, &after)
                .context("failed to save journal entry")?;
            changes.push(repo.file_for_entry(date));
        } else {
            println!("entry left unchanged");
        }

        if let Some(after_summary) = summary {
            if before_summary != after_summary {
                summaries.set(date, after_summary);

                repo.write_summaries(&summaries)
                    .context("failed to save summaries")?;
                changes.push(repo.file_for_summaries());
            }
        }

        // push changes if made
        if !changes.is_empty() {
            if let Some(git) = git {
                println!("committing new changes");

                git.commit(changes, &construct_commit_message(&repo.config.git.message, !exists, Some(date), force))?;
                git.push()?;
            }
        }
    } else {
        if void {
            println!("cannot preview note in void mode, continuing")
        } else {
            editor::preview(&content, &name, &repo.config.editor)?;
        }
    }

    Ok(())
}
