use anyhow::{anyhow, Context, Result};
use chrono::{Days, Local, NaiveDate};

/// performs some heuristics to determine whether an input string is absolute
/// we mean absolute in the sense of that a date parsed form this string cannot simply be shifted
pub fn is_absolute(input: &str) -> bool {

    // see https://docs.rs/fuzzydate/0.2.2/fuzzydate/
    let forbidden = vec![
        ".", "/", "-",
        "monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday",
        "mon", "tue", "wed", "thu", "fri", "sat", "sun",
        "january", "february", "march", "april", "may", "june", "july", "august", "september", "october", "november", "december",
        "jan", "feb", "mar", "apr", "jun", "jul", "aug", "sep", "oct", "nov", "dec"
    ];

    for pattern in forbidden {
        if input.contains(pattern) { return true; }
    }

    false
}

/// parses an absolute date from a string
pub fn parse_absolute(input: &str) -> Result<NaiveDate> {
    fuzzydate::parse(input)
        .map(|datetime| datetime.date())
        .context("could not parse the given date")
}

/// parses a relative date relative from `base` from a string
pub fn parse_relative(input: &str, base: NaiveDate) -> Result<NaiveDate> {
    if is_absolute(input) {
        return Err(anyhow!("cannot parse relative date, is absolute"));
    }

    let now = Local::now().naive_local().date();
    let date = parse_absolute(input)?;

    let difference = (date - now).num_days();

    if difference >= 0 {
        base.checked_add_days(Days::new(difference as u64))
    } else {
        base.checked_sub_days(Days::new(-difference as u64))
    }
        .context("failed to offset relative date")
}
