use std::{collections::HashMap, path::{Path, PathBuf}, process::{Command, Stdio}};

use agent::Agent;
use anyhow::{anyhow, Context, Result};
use chrono::{NaiveDate, NaiveDateTime};

use crate::repository::config::ConfigGitMessage;

mod agent;

pub struct Git {
    offline: bool,
    agent: Option<Agent>,
    repository: PathBuf
}

impl Git {

    /// creates a new git instance
    pub fn new(repository: &Path, key: Option<String>, offline: bool) -> Result<Self> {
        Ok(Self {
            offline,
            repository: repository.to_owned(),
            agent: key.map(|key| Agent::start(&key)).transpose()?
        })
    }

    /// run a git command
    fn run(&self, mut command: Command) -> Result<String> {
        command
            .current_dir(&self.repository)
            .stdin(Stdio::null());

        // apply agent if enabled
        if let Some(agent) = &self.agent {
            agent.apply(&mut command);
        }

        let output = command.output()
            .context("failed to execute git")?;

        if !output.status.success() {
            Err(anyhow!("{}", String::from_utf8_lossy(&output.stderr).trim().to_owned()))
                .context("git failed with non-zero exit code")
        } else { Ok(String::from_utf8_lossy(&output.stdout).to_string()) }
    }

    /// pull changes from the git remote
    pub fn pull(&self) -> Result<()> {
        if self.offline {
            println!("skipping repository pull in offline mode");
            return Ok(());
        }

        let mut command = Command::new("git");
        command.arg("pull");

        self.run(command)
            .context("failed to run git pull")?;

        Ok(())
    }

    /// make a commit on the repository
    pub fn commit(&self, files: Vec<PathBuf>, message: &str) -> Result<()>{
        // files my be potentially untracked
        let mut add = Command::new("git");
        add.arg("add").args(&files);

        self.run(add)
            .context("failed to run git add")?;

        let mut commit = Command::new("git");
        commit
            .arg("commit")
            .arg("-m").arg(message)
            .args(&files);

        self.run(commit)
            .context("failed to run git commit")?;

        Ok(())
    }

    /// push changes to the git remote
    pub fn push(&self) -> Result<()> {
        if self.offline {
            println!("skipping repository push in offline mode");
            return Ok(());
        }

        let mut command = Command::new("git");
        command.arg("push");

        self.run(command)
            .context("failed to run git push")?;

        Ok(())
    }

    /// retrieves the history of each file in the repository by parsing the git history
    pub fn history(&self, config: &ConfigGitMessage) -> Result<HashMap<String, Vec<HistoryCommit>>> {
        const DATE_FORMAT: &str = "%d.%m.%Y %H:%M:%S";

        let mut command = Command::new("git");
        command.arg("log")
            .arg(format!("--date=format:{DATE_FORMAT}"))
            .arg("--pretty=format:---commit---\n%ad\n%s")
            .arg("--name-only");

        let output = self.run(command)
            .context("failed to run git log")?;

        let mut map = HashMap::new();

        for commit in output.trim().split("---commit---\n") {
            if commit.trim().is_empty() { continue; }

            let lines = commit.trim().split("\n").collect::<Vec<_>>();

            if (lines.len() < 2) {
                return Err(anyhow!("git log contains invalid statement: {commit}"))
            }

            let commit = HistoryCommit {
                time: NaiveDateTime::parse_from_str(lines[0], DATE_FORMAT)
                    .context(format!("git log contains invalid date: {}", lines[0]))?,
                force: lines[1].contains(&config.tag_force)
            };

            for file in lines.iter().skip(2) {
                map.entry(file.to_string())
                    .or_insert(vec!())
                    .push(commit.clone());
            }
        }

        Ok(map)
    }

}

#[derive(Clone)]
pub struct HistoryCommit {
    /// time of the commit
    pub time: NaiveDateTime,
    /// whether the commit was forced (in the sense of being out of permitted time slot)
    pub force: bool
}

pub fn construct_commit_message(config: &ConfigGitMessage, new: bool, date: Option<NaiveDate>, force: bool) -> String {

    // verb: what tags
    format!("{}: {} {}",
        if new { &config.verb_new } else { &config.verb_update },
        if let Some(date) = date { date.format(&config.date_format).to_string() } else { config.named.clone() },
        if force { config.tag_force.as_str() } else { "" }
    ).trim().to_string()
}
