use std::{fs, io::{Read, Write}, path::PathBuf, process::{Child, Command, Stdio}};

use anyhow::{anyhow, Context, Result};
use chrono::Local;

pub struct Agent {
    agent: Child,
    address: PathBuf,
}

impl Agent {
    /// start the agent and load an ssh key
    pub fn start(key: &str) -> Result<Self> {
        let address = PathBuf::from("/tmp")
                .join(format!("me-agent-{}.sock", Local::now().timestamp_millis()));

        // spawn agent
        let mut agent = Command::new("ssh-agent")
            .arg("-a").arg(address.as_os_str()) // set bind address
            .arg("-D") // don't fork
            .stdin(Stdio::null())
            .stdout(Stdio::piped())
            .stderr(Stdio::null())
            .spawn()
            .context("failed to spawn ssh agent")?;

        // wait for agent to print stuff
        let mut buffer = vec![0u8; 16];
        if matches!(agent.stdout.take().expect("stdout piped").read(&mut buffer), Ok(0) | Err(_)) {
            return Err(anyhow!("ssh-agent failed to start properly, did not print anything to stdout"));
        }

        let instance = Agent { agent, address };

        // load key
        let mut add = Command::new("ssh-add");
        add.arg("-q").arg("-"); // shut up and read from stdin
        add.stdin(Stdio::piped()).stdout(Stdio::piped()).stderr(Stdio::piped());
        instance.apply(&mut add);

        let mut child = add.spawn()
            .context("failed to spawn ssh-add")?;

        // write data to stdin
        let mut stdin = child.stdin.take().expect("stdin is piped");
        stdin.write_all(key.as_bytes()).context("failed to write to ssh-add stdin")?;
        if !key.ends_with("\n") { // we need to have it end with a newline otherwise ssh-agent won't accept it
            stdin.write_all("\n".as_bytes()).context("failed to write to ssh-add stdin")?;
        }

        stdin.flush().context("failed to flush ssh-add stdin")?;
        drop(stdin); // close the stdin

        let output = child.wait_with_output().context("failed to wait for ssh-add")?;

        if !output.status.success() {
            return Err(anyhow!("{}", String::from_utf8_lossy(&output.stderr).trim().to_owned()))
                .context("failed to use ssh-add")
        }

        Ok(instance)
    }

    /// apply the agent to be used by a command
    pub fn apply(&self, command: &mut Command) {
        command.env("SSH_AUTH_SOCK", self.address.as_os_str());
    }

}

impl Drop for Agent {
    fn drop(&mut self) {
        // these are all non-critical, so we can accept if they fail
        self.agent.kill().ok();
        fs::remove_file(&self.address).ok();
    }
}
