use std::{fs::{self, OpenOptions}, io::Write, os::unix::fs::OpenOptionsExt, path::{Path, PathBuf}, process::Command};

use anyhow::{Context, Result};
use chrono::Local;

use crate::repository::config::ConfigEditor;

/// creates a temporary file for editing
fn start_tmp(content: &str, name: &str, readonly: bool) -> Result<PathBuf> {
    let parent = PathBuf::from("/tmp")
        .join(format!("me-{}", Local::now().timestamp_millis()));

    fs::create_dir_all(&parent)
        .context("failed to create temporary directory")?;

    let file = parent.join(name);

    let mut options = OpenOptions::new();
    options.create_new(true);
    options.write(true);

    options.mode(
        if readonly { 0o0400 }
        else { 0o0600 }
    );

    let mut handle = options.open(&file)
        .context("failed to create temporary file")?;

    handle.write(content.as_bytes())
        .context("failed to write to file")?;

    Ok(file)
}

/// removes the temporary file after editing
fn end_tmp(path: &Path) -> Result<String> {
    let content = fs::read_to_string(path)
        .context("failed to read again from temporary file")?;

    fs::remove_dir_all(
        path.parent()
            .context("temporary directory did not have parent (bug this)")?
    ).context("failed to remove temporary directory")?;

    Ok(content)
}

/// opens the editor
fn start_editor(editor: &str, args: &Vec<String>, path: &Path) -> Result<()> {
    let mut command = Command::new(editor);
    command.args(args);
    command.arg(path);

    let mut child = command.spawn()
        .context("failed to spawn editor command")?;

    child.wait()
        .context("failed to wait for editor command to exit")?;

    Ok(())
}

/// opens something in the editor
fn open(content: &str, name: &str, editor: &str, args: &Vec<String>, readonly: bool) -> Result<String> {
    let path = start_tmp(content, name, readonly)
        .context("failed to prepare for editor")?;

    let result = start_editor(editor, args, &path); // don't process the result yet

    let next = end_tmp(&path)
        .context("failed to clean up after editor (THIS MIGHT LEAK YOUR DATA)")?;

    result.map(|_| next)
}

pub fn edit(content: &str, name: &str, editor: &ConfigEditor) -> Result<String> {
    open(content, name, &editor.command, &editor.arguments,  false)
}

pub fn preview(content: &str, name: &str, editor: &ConfigEditor) -> Result<()> {
    open(content, name, &editor.command, &editor.arguments, true).map(|_| ())
}
