use chrono::{Local, NaiveDate};
use crossterm::event::{self, Event, KeyCode, KeyEvent, KeyEventKind};
use list::{BrowseEntry, BrowseEntryList};
use ratatui::{layout::Margin, style::{Style, Stylize}, text::ToLine, widgets::{Block, Padding}, DefaultTerminal, Frame};
use anyhow::Result;

pub mod list;

pub fn select(entries: &[BrowseEntry], initial_selection: Option<NaiveDate>) -> Result<Option<NaiveDate>> {
    let terminal = ratatui::init();
    let result = App::new(entries, initial_selection).run(terminal);
    ratatui::restore();

    result
}

struct App<'a> {
    alive: bool,
    list: BrowseEntryList<'a>
}

impl<'a> App<'a> {
    pub fn new(entries: &'a [BrowseEntry], initial_selection: Option<NaiveDate>) -> Self {
        let mut list = BrowseEntryList::new(entries);
        if let Some(init) = initial_selection { list.selection_absolute(init); }

        Self {
            list,
            alive: true
        }
    }

    fn run(&mut self, mut terminal: DefaultTerminal) -> Result<Option<NaiveDate>> {
        while self.alive {
            terminal.draw(|f| self.render(f))?;

            match event::read()? {
                Event::Key(key_event) if key_event.kind == KeyEventKind::Press => {
                    if let Some(date) = self.handle_key(key_event) {
                        return Ok(Some(date));
                    }
                }
                _ => {}
            }
       }

       Ok(None)
    }

    fn handle_key(&mut self, event: KeyEvent) -> Option<NaiveDate> {
        match event.code {
            KeyCode::Char('q') | KeyCode::Esc => self.alive = false,

            // move one day
            KeyCode::Up | KeyCode::Char('k') => self.list.selection_relative(-1),
            KeyCode::Down | KeyCode::Char('j') => self.list.selection_relative(1),

            // move a week
            KeyCode::Char('w') => self.list.selection_relative(7),
            KeyCode::Char('b') => self.list.selection_relative(-7),

            // move to the top and bottom (yes, this is hack but oh well)
            KeyCode::Char('g') => self.list.selection_relative(i32::MIN / 2),
            KeyCode::Char('G') => self.list.selection_relative(i32::MAX / 2),

            // move to today
            KeyCode::Char('f') => self.list.selection_absolute(Local::now().date_naive()),

            // user has selected
            KeyCode::Enter => return Some(self.list.selection()),

            _ => {}
        }

        None
    }

    fn render(&self, frame: &mut Frame) {
        let today = Local::now().date_naive().format("%d.%m.%y").to_string();
        let total = format!("{} entries", self.list.total());

        let block = Block::bordered()
            .title("me journal".to_line().style(Style::default().italic()).left_aligned())
            .title(today.to_line().centered())
            .title(total.to_line().right_aligned())
            .title_bottom("<j>: down, <k>: up, <enter>: open entry, <q>: quit".to_line().style(Style::default().dim()).centered())
            .padding(Padding::new(1, 1, 0, 0));

        let area = frame.area().inner(Margin::new(2, 1));
        frame.render_widget(&block, area);
        frame.render_widget(&self.list, block.inner(area));
    }
}
