
use chrono::{NaiveDate, NaiveTime};
use ratatui::style::{Color, Style, Styled, Stylize};
use ratatui::text::ToLine;
use ratatui::widgets::{Gauge, Widget};
use ratatui::prelude::{Rect, Buffer};


pub struct BrowseEntry {
    /// date of the entry
    pub date: NaiveDate,
    /// summary of the entry if provided
    pub summary: Option<String>,
    /// history of the entry if provided
    pub history: Option<EntryHistory>,
    /// size of the entry on disk if exists
    pub size: Option<u64>
}

pub struct EntryHistory {
    /// last time the entry was edited (non-forced)
    pub time: Option<NaiveTime>,
    /// amount of changes the entry has had
    pub changes: usize,
    /// whether the entry was forced once
    pub force: bool
}

pub struct BrowseEntryList<'a> {
    entries: &'a [BrowseEntry],
    selected: usize,

    max_changes: usize,
    max_size: u64
}

impl<'a> BrowseEntryList<'a> {
    pub fn new(entries: &'a [BrowseEntry]) -> Self {
        Self {
            selected: entries.len() - 1,
            max_changes: entries.iter().filter_map(|e| e.history.as_ref()).map(|h| h.changes).max().unwrap_or(1),
            max_size: *entries.iter().filter_map(|e| e.size.as_ref()).max().unwrap_or(&1),
            entries,
        }
    }

    pub fn selection_relative(&mut self, offset: i32) {
        self.selected = (self.selected as i32 + offset).clamp(0, self.entries.len() as i32 - 1) as usize;
    }

    pub fn selection_absolute(&mut self, date: NaiveDate) {
        if let Some((i, _)) = self.entries.iter().enumerate().find(|(_, entry)| entry.date == date) {
            self.selected = i;
        }
    }

    pub fn render_entry(&self, item: usize, area: Rect, buf: &mut Buffer) {
        let entry = &self.entries[item];
        let selected = self.selected == item;

        let weekday = entry.date.format("%a").to_string();
        let weekday = if weekday == "Sat" || weekday == "Sun" {
            weekday.to_line().style(Style::default().red())
        } else {
            weekday.to_line()
        };

        let date = entry.date.format("%d.%m.%y").to_string();
        let date = if selected {
            date.to_line().style(Style::default().black().on_white())
        } else {
            date.to_line()
        };

        weekday.render(Rect::new(area.x, area.y, 3, area.height), buf);
        date.render(Rect::new(area.x + 5, area.y, 8, area.height), buf);

        let contents = entry.summary.as_ref()
            .map(|s| s.to_line())
            .unwrap_or_else(|| "no summary provided".to_line().set_style(Style::default().italic().dim()));

        contents.render(Rect::new(area.x + 15, area.y, area.width - 15 - 15, area.height), buf);

        // render file size if it exists
        if let Some(size) = entry.size {
            get_progress_bar(size as f32 / self.max_size as f32, 8)
                .to_line()
                .set_style(Style::default().dark_gray())
                .render(Rect::new(area.x + area.width - 13, area.y, 8, area.height), buf);
        }

        // render history if there is one
        if let Some(ref history) = entry.history {

            let amount = format!("{:02}", history.changes);
            amount.to_line()
                .set_style(Style::default().fg(get_amount_color((history.changes - 1) as f32 / (self.max_changes - 1) as f32)))
                .render(Rect::new(area.x + area.width - 3, area.y, 2, area.height), buf);

            if (history.force) {
                "*".to_line()
                    .set_style(Style::default().dim())
                    .render(Rect::new(area.x + area.width - 1, area.y, 1, area.height), buf);
            }
        }
    }

    pub fn selection(&self) -> NaiveDate {
        self.entries[self.selected].date
    }

    pub fn total(&self) -> usize {
        self.entries.len()
    }

    pub fn edits(&self) -> usize {
        self.entries.iter()
            .filter_map(|a| a.history.as_ref())
            .map(|h| h.changes)
            .sum()
    }
}

impl Widget for &BrowseEntryList<'_> {
    fn render(self, area: Rect, buf: &mut Buffer) {
        let skip = if area.height as usize > self.entries.len() {
            -(area.height as i32 - self.entries.len() as i32)
        } else {
            (self.selected as i32 - area.height as i32 / 2).clamp(0, self.entries.len() as i32 - area.height as i32)
        };

        for i in 0..area.height as usize {
            let item = i as i32 + skip;
            if item < 0 || item >= self.entries.len() as i32 { continue; }

            self.render_entry(item as usize, Rect::new(area.x, area.y + i as u16, area.width, 1), buf);
        }
    }
}

/// returns the color for a given amount percentage by using lerp
/// so yes, it will look inherently incorrect probably
pub fn get_amount_color(amount: f32) -> Color {

    // yes these colors are shamelessly adapted from my terminal color scheme
    const LOW: (u8, u8, u8) = (216u8, 216u8, 216u8);
    const HIGH: (u8, u8, u8) = (122u8, 166u8, 218u8);

    Color::Rgb(
        (LOW.0 as f32 + (HIGH.0 as f32 - LOW.0 as f32) * amount) as u8,
        (LOW.1 as f32 + (HIGH.1 as f32 - LOW.1 as f32) * amount) as u8,
        (LOW.2 as f32 + (HIGH.2 as f32 - LOW.2 as f32) * amount) as u8,
    )
}

pub fn get_progress_bar(amount: f32, length: usize) -> String {
    let empty = 1f32 - amount;

    (0..length)
        .map(|i| ((empty - (i as f32 / length as f32)) / (1f32 / length as f32) * 9f32) as usize)
        .map(|c| 8 - c.clamp(0, 8))
        .map(|c|  match c {
            0 => ' ',
            1 => '▕',
            2 => '▕',
            3 => '▐',
            4 => '▐',
            5 => '▐',
            6 => '█',
            7 => '█',
            8 => '█',
            _ => '?'
        })
        .collect()
}
